/*
 * Copyright (c) Doug Palmer <doug@charvolant.org> 2005
 *
 * See LICENSE for licensing details.
 * 
 * $Id$
 */

package org.charvolant.sudoku.gui;

import java.awt.ComponentOrientation;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Panel;

import org.charvolant.sudoku.Board;
import org.charvolant.sudoku.Cell;
import org.charvolant.sudoku.ModelEvent;
import org.charvolant.sudoku.ModelListener;

/**
 * A panel that displays a full board.
 * 
 * @author doug
 */
public class BoardPanel extends Panel implements ModelListener {
  /** The border inset for cells */
  private static final int BORDER = 1;
  
  /** The underlying board */
  private Board board;
  /** The display preferences */
  private DisplayPreferences preferences;
  
  /**
   * Create a new panel for this board.
   */
  public BoardPanel(Board board) {
    this.board = board;
    this.board.addListener(this);
    this.preferences = new DisplayPreferences();
    this.preferences.addListener(this);
    this.initialize();
  }
  
  /**
   * Initialise the board.
   * <p>
   * Get all the cells, sort them into Y-X order
   * and then add them to a grid layout.
   *
   */
  public void initialize() {
    int size = this.board.getSize();
    int sq = this.board.getSquareSize();
    Cell[][] cells = this.board.getCells();
    Insets insets;
    int i, j;
    
    this.setLayout(new GridBagLayout());
    this.setComponentOrientation(ComponentOrientation.LEFT_TO_RIGHT);
    for (i = 0; i < size; i++)
      for (j = 0; j < size; j++) {
        insets = new Insets(
            i != 0 && i % sq == 0 ? this.BORDER : 0,
            j != 0 && j % sq == 0 ? this.BORDER : 0,
            i % sq == (sq - 1) ? this.BORDER : 0,
            j % sq == (sq - 1) ? this.BORDER : 0
        );
        this.add(
            new CellCanvas(cells[i][j], this.preferences), 
            new GridBagConstraints(
                j, i,
                1, 1,
                1.0, 1.0,
                GridBagConstraints.CENTER,
                GridBagConstraints.BOTH,
                insets,
                0, 0
            )
        );
      }
  }
  
  /**
   * Receive a change in the model.
   *
   * @param event The change.
   *
   * @see org.charvolant.sudoku.ModelListener#modelUpdate(org.charvolant.sudoku.ModelEvent)
   */
  public void modelUpdate(ModelEvent event) {
    this.repaint();
  }
}
